/**
 * Centronium Extension — Popup Logic (CSP-compliant, no inline handlers)
 */

let wallet = null;
let balances = null;
let price = 0;
let currentSeed = '';

// ─── BIP39-lite word list ────────────────────────────────────
const WL=['abandon','ability','able','about','above','absent','absorb','abstract','absurd','abuse','access','accident','account','accuse','achieve','acid','acoustic','acquire','across','act','action','actor','actress','actual','adapt','add','addict','address','adjust','admit','adult','advance','advice','aerobic','affair','afford','afraid','again','age','agent','agree','ahead','aim','air','airport','aisle','alarm','album','alcohol','alert','alien','all','alley','allow','almost','alone','alpha','already','also','alter','always','amateur','amazing','among','amount','amused','analyst','anchor','ancient','anger','angle','angry','animal','ankle','announce','annual','another','answer','antenna','antique','anxiety','any','apart','apology','appear','apple','approve','april','arch','arctic','area','arena','argue','arm','armed','armor','army','around','arrange','arrest','arrive','arrow','art','artefact','artist','artwork','ask','aspect','assault','asset','assist','assume','asthma','athlete','atom','attack','attend','attitude','attract','auction','audit','august','aunt','author','auto','autumn','average','avocado','avoid','awake','aware','awesome','awful','awkward','axis','baby','bachelor','bacon','badge','bag','balance','balcony','ball','bamboo','banana','banner','bar','barely','bargain','barrel','base','basic','basket','battle','beach','bean','beauty','because','become','beef','before','begin','behave','behind','believe','below','belt','bench','benefit','best','betray','better','between','beyond','bicycle','bid','bike','bind','biology','bird','birth','bitter','black','blade','blame','blanket','blast','bleak','bless','blind','blood','blossom','blow','blue','blur','blush','board','boat','body','boil','bomb','bone','bonus','book','boost','border','boring','borrow','boss','bottom','bounce','box','boy','bracket','brain','brand','brass','brave','bread','breeze','brick','bridge','brief','bright','bring','brisk','broccoli','broken','bronze','broom','brother','brown','brush','bubble','buddy','budget','buffalo','build','bulb','bulk','bullet','bundle','bunny','burden','burger','burst','bus','business','busy','butter','buyer','buzz','cabbage','cabin','cable','cactus','cage','cake','call','calm','camera','camp','can','canal','cancel','candy','cannon','canoe','canvas','canyon','capable','capital','captain','car','carbon','card','cargo','carpet','carry','cart','case','cash','casino','castle','casual','cat','catalog','catch','category','cattle','caught','cause','caution','cave','ceiling','celery','cement','census','century','cereal','certain','chair','chalk','champion','change','chaos','chapter','charge','chase','cheap','check','cheese','chef','cherry','chest','chicken','chief','child','chimney','choice','choose','chronic','chuckle','chunk','churn','citizen','city','civil','claim','clap','clarify','claw','clay','clean','clerk','clever','click','client','cliff','climb','clinic','clip','clock','clog','close','cloth','cloud','clown','club','clump','cluster','clutch','coach','coast','coconut','code','coffee','coil','coin','collect','color','column','combine','come','comfort','comic','common','company','concert','conduct','confirm','congress','connect','consider','control','convince','cook','cool','copper','copy','coral','core','corn','correct','cost','cotton','couch','country','couple','course','cousin','cover','coyote','crack','cradle','craft','cram','crane','crash','crater','crawl','crazy','cream','credit','creek','crew','cricket','crime','crisp','critic','crop','cross','crouch','crowd','crucial','cruel','cruise','crumble','crush','cry','crystal','cube','culture','cup','cupboard','curious','current','curtain','curve','cushion','custom','cute','cycle','dad','damage','damp','dance','danger','daring','dash','daughter','dawn','day','deal','debate','debris','decade','december','decide','decline','decorate','decrease','deer','defense','define','defy','degree','delay','deliver','demand','demise','denial','dentist','deny','depart','depend','deposit','depth','deputy','derive','describe','desert','design','desk','despair','destroy','detail','detect','develop','device','devote','diagram','dial','diamond','diary','dice','diesel','diet','differ','digital','dignity','dilemma','dinner','dinosaur','direct','dirt','disagree','discover','disease','dish','dismiss','disorder','display','distance','divert','divide','divorce','dizzy','doctor','document','dog','doll','dolphin','domain','donate','donkey','donor','door','dose','double','dove','draft','dragon','drama','drastic','draw','dream','dress','drift','drill','drink','drip','drive','drop','drum','dry','duck','dumb','dune','during'];

function genSeed() {
  const w = [];
  for (let i = 0; i < 12; i++) {
    const idx = crypto.getRandomValues(new Uint16Array(1))[0] % WL.length;
    w.push(WL[idx]);
  }
  return w.join(' ');
}

async function seedToAddr(seed) {
  const enc = new TextEncoder();
  const hash = await crypto.subtle.digest('SHA-256', enc.encode(seed + ':centronium:v1'));
  const arr = new Uint8Array(hash);
  const hex = Array.from(arr).map(b => b.toString(16).padStart(2, '0')).join('');
  return 'CENTRO_' + hex.substring(0, 40);
}

// ─── Screen Navigation ───────────────────────────────────────
function showScreen(name) {
  document.querySelectorAll('.screen').forEach(s => s.classList.remove('active'));
  const el = document.getElementById('screen-' + name);
  if (el) el.classList.add('active');
  const hdr = document.getElementById('mainHeader');
  hdr.style.display = (name === 'onboard' || name === 'seed') ? 'none' : 'flex';
  if (name === 'main') refreshData();
  if (name === 'receive') document.getElementById('recvAddr').textContent = wallet ? wallet.address : '—';
  if (name === 'send') updateAvail();
  if (name === 'settings') document.getElementById('setAddr').textContent = wallet ? wallet.address : '—';
}

// ─── Wallet Creation ─────────────────────────────────────────
async function createWallet() {
  const seed = genSeed();
  const addr = await seedToAddr(seed);
  wallet = { address: addr, seed: seed };
  currentSeed = seed;
  chrome.runtime.sendMessage({ type: 'CREATE_WALLET', seed: seed, address: addr });
  renderSeed(seed);
  showScreen('seed');
}

async function importWallet() {
  const seed = prompt('Enter your 12-word recovery phrase:');
  if (!seed || seed.trim().split(/\s+/).length < 12) return toast('Invalid phrase', 'err');
  const trimmed = seed.trim();
  const addr = await seedToAddr(trimmed);
  wallet = { address: addr, seed: trimmed };
  currentSeed = trimmed;
  chrome.runtime.sendMessage({ type: 'IMPORT_WALLET', seed: trimmed, address: addr });
  showScreen('main');
  toast('Wallet imported!', 'ok');
}

function renderSeed(seed) {
  const grid = document.getElementById('seedGrid');
  grid.innerHTML = '';
  seed.split(' ').forEach(function(w, i) {
    const div = document.createElement('div');
    div.className = 'sw';
    const numSpan = document.createElement('span');
    numSpan.className = 'sn';
    numSpan.textContent = i + 1;
    div.appendChild(numSpan);
    div.appendChild(document.createTextNode(w));
    grid.appendChild(div);
  });
}

// ─── Data Refresh ────────────────────────────────────────────
async function refreshData() {
  if (!wallet) return;
  try {
    const results = await Promise.all([
      chrome.runtime.sendMessage({ type: 'GET_BALANCE' }),
      chrome.runtime.sendMessage({ type: 'GET_PRICE' }),
      chrome.runtime.sendMessage({ type: 'GET_TXS' }),
    ]);
    var balRes = results[0], priceRes = results[1], txRes = results[2];
    balances = balRes;
    price = (priceRes && priceRes.price) ? priceRes.price : 0;
    var c = (balRes && balRes.centro) ? balRes.centro : 0;
    var u = (balRes && balRes.cusdc) ? balRes.cusdc : 0;
    document.getElementById('balAmount').textContent = c.toFixed(4);
    document.getElementById('balUsd').textContent = '\u2248 $' + (c * price + u).toFixed(2) + ' USD';
    document.getElementById('bCentro').textContent = c.toFixed(2);
    document.getElementById('bCusdc').textContent = u.toFixed(2);
    renderTxs((txRes && txRes.transactions) ? txRes.transactions : []);
  } catch (e) { console.error(e); }
}

function renderTxs(txs) {
  var addr = wallet ? wallet.address : '';
  var container = document.getElementById('txListEl');
  container.innerHTML = '';

  if (!txs.length) {
    container.innerHTML = '<div style="padding:30px;text-align:center;color:var(--tx3);font-size:12px">No transactions yet</div>';
    return;
  }

  txs.slice(0, 8).forEach(function(tx) {
    var isIn = tx.to_address === addr && tx.from_address !== addr;
    var isRw = ['mining_reward', 'block_reward', 'signup_bonus'].indexOf(tx.tx_type) !== -1;
    var ic = 'out', icon = '\u2B06', label = 'Sent', cls = 'red', sign = '-';
    if (isRw) { ic = 'rw'; icon = '\u26CF'; label = 'Reward'; cls = 'grn'; sign = '+'; }
    else if (isIn) { ic = 'in'; icon = '\u2B07'; label = 'Received'; cls = 'grn'; sign = '+'; }
    var other = isIn ? tx.from_address : tx.to_address;
    var short = other ? (other.substring(0, 10) + '\u2026' + other.slice(-4)) : '\u2014';

    var item = document.createElement('div');
    item.className = 'tx-item';
    item.setAttribute('data-txhash', tx.tx_hash);
    item.innerHTML =
      '<div class="tx-ic ' + ic + '">' + icon + '</div>' +
      '<div class="tx-info"><div class="tt">' + label + '</div><div class="ta">' + short + '</div></div>' +
      '<div class="tx-right"><div class="tam ' + cls + '">' + sign + parseFloat(tx.amount).toFixed(4) + '</div><div class="ttm">' + tx.coin_symbol + '</div></div>';
    item.addEventListener('click', function() {
      chrome.tabs.create({ url: 'https://centroscan.com/tx/' + tx.tx_hash });
    });
    container.appendChild(item);
  });
}

// ─── Send ────────────────────────────────────────────────────
function updateAvail() {
  var coin = document.getElementById('sCoin').value;
  var bal = coin === 'cUSDC' ? ((balances && balances.cusdc) || 0) : ((balances && balances.centro) || 0);
  document.getElementById('sAvail').textContent = bal.toFixed(4) + ' ' + coin;
}

function setMax() {
  var coin = document.getElementById('sCoin').value;
  var bal = coin === 'cUSDC' ? ((balances && balances.cusdc) || 0) : Math.max(0, ((balances && balances.centro) || 0) - 0.001);
  document.getElementById('sAmt').value = bal;
}

async function doSend() {
  var to = document.getElementById('sTo').value.trim();
  var amt = parseFloat(document.getElementById('sAmt').value);
  var coin = document.getElementById('sCoin').value;
  var memo = document.getElementById('sMemo').value;
  if (!to || to.indexOf('CENTRO_') !== 0) return toast('Invalid address', 'err');
  if (!amt || amt <= 0) return toast('Invalid amount', 'err');
  var res = await chrome.runtime.sendMessage({ type: 'SEND_TX', to: to, amount: amt, coin: coin, memo: memo });
  if (res && res.success) {
    toast('Transaction sent!', 'ok');
    setTimeout(function() { showScreen('main'); }, 1500);
  } else {
    toast((res && res.error) || 'Send failed', 'err');
  }
}

// ─── Clipboard ───────────────────────────────────────────────
function copyAddr() {
  if (!wallet) return;
  navigator.clipboard.writeText(wallet.address);
  toast('Copied!', 'ok');
}

function copySeed() {
  if (currentSeed) {
    navigator.clipboard.writeText(currentSeed);
    toast('Copied!', 'ok');
  }
}

// ─── Settings ────────────────────────────────────────────────
async function showSeed() {
  var res = await chrome.runtime.sendMessage({ type: 'GET_SEED' });
  if (res && res.seed) {
    currentSeed = res.seed;
    renderSeed(res.seed);
    showScreen('seed');
  } else {
    toast('No seed available', 'err');
  }
}

function lockWallet() {
  chrome.runtime.sendMessage({ type: 'LOCK' });
  wallet = null;
  balances = null;
  showScreen('onboard');
  toast('Wallet locked', 'ok');
}

async function removeWallet() {
  if (!confirm('Remove wallet? Make sure you have your recovery phrase!')) return;
  await chrome.runtime.sendMessage({ type: 'REMOVE_WALLET' });
  wallet = null;
  balances = null;
  currentSeed = '';
  showScreen('onboard');
  toast('Wallet removed', 'ok');
}

// ─── Toast ───────────────────────────────────────────────────
function toast(msg, type) {
  type = type || 'ok';
  var t = document.getElementById('toast');
  t.textContent = msg;
  t.className = 'toast show ' + type;
  setTimeout(function() { t.className = 'toast'; }, 2500);
}

// ─── Event Binding (CSP-compliant, no inline handlers) ───────
document.addEventListener('DOMContentLoaded', function() {

  // Bind all data-action clicks via event delegation
  document.body.addEventListener('click', function(e) {
    var target = e.target.closest('[data-action]');
    if (!target) return;
    var action = target.getAttribute('data-action');

    // Screen navigation: data-action="screen-{name}"
    if (action.indexOf('screen-') === 0) {
      showScreen(action.replace('screen-', ''));
      return;
    }

    switch (action) {
      case 'create-wallet':  createWallet(); break;
      case 'import-wallet':  importWallet(); break;
      case 'copy-seed':      copySeed(); break;
      case 'copy-addr':      copyAddr(); break;
      case 'set-max':        setMax(); break;
      case 'do-send':        doSend(); break;
      case 'show-seed':      showSeed(); break;
      case 'lock-wallet':    lockWallet(); break;
      case 'remove-wallet':  removeWallet(); break;
      case 'open-explorer':
        chrome.tabs.create({ url: 'https://centroscan.com' });
        break;
    }
  });

  // Token select change
  var sCoin = document.getElementById('sCoin');
  if (sCoin) {
    sCoin.addEventListener('change', updateAvail);
  }

  // ─── Init: check for existing wallet ────────────────────
  chrome.runtime.sendMessage({ type: 'UNLOCK' }, function(res) {
    if (res && res.address) {
      wallet = { address: res.address };
      showScreen('main');
    } else {
      showScreen('onboard');
    }
  });
});
