/**
 * Centronium Provider — window.centronium
 * Injected into web pages by the Centronium Wallet extension.
 * Provides a provider API similar to window.ethereum for dApp integration.
 */

(function () {
  let requestId = 0;
  const pendingRequests = new Map();

  const provider = {
    isCentronium: true,
    chainId: '0x12f8e', // 77710
    networkVersion: '77710',
    selectedAddress: null,

    async request(args) {
      const { method, params } = args;

      switch (method) {
        case 'centro_requestAccounts':
        case 'eth_requestAccounts': {
          const res = await sendToExtension({ type: 'GET_STATE' });
          if (res.address) {
            provider.selectedAddress = res.address;
            return [res.address];
          }
          throw new Error('No wallet connected');
        }

        case 'centro_getBalance': {
          return sendToExtension({ type: 'GET_BALANCE' });
        }

        case 'centro_sendTransaction': {
          return sendToExtension({ type: 'SEND_TX', ...params });
        }

        case 'eth_chainId':
          return '0x12f8e';

        case 'net_version':
          return '77710';

        case 'eth_accounts': {
          const res = await sendToExtension({ type: 'GET_STATE' });
          return res.address ? [res.address] : [];
        }

        default:
          throw new Error(`Unsupported method: ${method}`);
      }
    },

    on(event, callback) {
      window.addEventListener('message', (e) => {
        if (e.data?.target !== 'centronium-page' || !e.data.event) return;
        if (e.data.event.type === 'ACCOUNT_CHANGED' && event === 'accountsChanged') {
          callback([e.data.event.address]);
        }
        if (e.data.event.type === 'NETWORK_CHANGED' && event === 'chainChanged') {
          callback(e.data.event.chainId);
        }
      });
    },

    removeListener() { /* stub */ },

    isConnected() { return true; },
  };

  function sendToExtension(payload) {
    return new Promise((resolve, reject) => {
      const id = ++requestId;
      pendingRequests.set(id, { resolve, reject });

      window.postMessage({
        target: 'centronium-extension',
        id,
        payload,
      }, '*');

      // Timeout
      setTimeout(() => {
        if (pendingRequests.has(id)) {
          pendingRequests.delete(id);
          reject(new Error('Extension request timeout'));
        }
      }, 30000);
    });
  }

  // Listen for responses
  window.addEventListener('message', (event) => {
    if (event.data?.target !== 'centronium-page') return;
    if (event.data.id && pendingRequests.has(event.data.id)) {
      const { resolve, reject } = pendingRequests.get(event.data.id);
      pendingRequests.delete(event.data.id);
      if (event.data.response?.error) {
        reject(new Error(event.data.response.error));
      } else {
        resolve(event.data.response);
      }
    }
  });

  // Expose provider
  window.centronium = provider;

  // Dispatch event for dApps
  window.dispatchEvent(new Event('centronium#initialized'));
})();
