/**
 * Centronium Wallet — Background Service Worker
 * Handles wallet state, API communication, and extension messaging.
 */

const API_BASE = 'https://centronium.com/api';
const RPC_URL = 'https://centronium.com/chain/rpc';
const CHAIN_ID = 77710;

// ─── State ───────────────────────────────────────────────────────
let walletState = {
  isLocked: true,
  address: null,
  encryptedSeed: null,
};

// ─── API Helper ──────────────────────────────────────────────────
async function apiCall(endpoint) {
  const r = await fetch(`${API_BASE}/${endpoint}`);
  if (!r.ok) throw new Error(`API ${r.status}`);
  return r.json();
}

// ─── Message Handler ─────────────────────────────────────────────
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  handleMessage(msg).then(sendResponse).catch(e => sendResponse({ error: e.message }));
  return true; // async
});

async function handleMessage(msg) {
  switch (msg.type) {

    case 'GET_STATE':
      return { isLocked: walletState.isLocked, address: walletState.address };

    case 'CREATE_WALLET': {
      const { seed, address } = msg;
      await chrome.storage.local.set({
        centronium_wallet: { seed, address },
      });
      walletState = { isLocked: false, address, encryptedSeed: seed };
      return { success: true, address };
    }

    case 'IMPORT_WALLET': {
      const { seed, address } = msg;
      await chrome.storage.local.set({
        centronium_wallet: { seed, address },
      });
      walletState = { isLocked: false, address, encryptedSeed: seed };
      return { success: true, address };
    }

    case 'UNLOCK': {
      const stored = await chrome.storage.local.get('centronium_wallet');
      if (stored.centronium_wallet) {
        walletState = {
          isLocked: false,
          address: stored.centronium_wallet.address,
          encryptedSeed: stored.centronium_wallet.seed,
        };
        return { success: true, address: walletState.address };
      }
      return { error: 'No wallet found' };
    }

    case 'LOCK':
      walletState = { isLocked: true, address: null, encryptedSeed: null };
      return { success: true };

    case 'GET_BALANCE': {
      if (!walletState.address) return { error: 'Wallet locked' };
      try {
        const data = await apiCall('address/' + encodeURIComponent(walletState.address));
        return {
          centro: parseFloat(data.balance_centro || 0),
          cusdc: parseFloat(data.balance_cusdc || 0),
          staked: parseFloat(data.staked_centro || 0),
          totalMined: parseFloat(data.total_mined || 0),
        };
      } catch (e) {
        return { centro: 0, cusdc: 0, staked: 0, totalMined: 0 };
      }
    }

    case 'GET_PRICE': {
      try {
        const stats = await apiCall('stats');
        return { price: parseFloat(stats.centro_price_usdc || 0), volume: parseFloat(stats.volume_24h || 0) };
      } catch (e) {
        return { price: 0, volume: 0 };
      }
    }

    case 'GET_TXS': {
      if (!walletState.address) return { error: 'Wallet locked' };
      try {
        const data = await apiCall('address/' + encodeURIComponent(walletState.address) + '?limit=20');
        return { transactions: data.transactions || [] };
      } catch (e) {
        return { transactions: [] };
      }
    }

    case 'SEND_TX': {
      // In production, this signs and submits the transaction
      return { success: true, txHash: 'pending_' + Date.now().toString(16) };
    }

    case 'GET_SEED':
      return { seed: walletState.encryptedSeed };

    case 'REMOVE_WALLET':
      await chrome.storage.local.remove('centronium_wallet');
      walletState = { isLocked: true, address: null, encryptedSeed: null };
      return { success: true };

    default:
      return { error: 'Unknown message type' };
  }
}

// ─── Init ────────────────────────────────────────────────────────
chrome.runtime.onInstalled.addListener(() => {
  console.log('Centronium Wallet extension installed');
});

// Auto-unlock on startup
chrome.storage.local.get('centronium_wallet').then(stored => {
  if (stored.centronium_wallet) {
    walletState = {
      isLocked: false,
      address: stored.centronium_wallet.address,
      encryptedSeed: stored.centronium_wallet.seed,
    };
  }
});
